
require(plyr)

f_simulation <- function(input_temp){

    simName <- paste(input_temp$Name, input_temp$Asset_Type, sep = '-')
  
	cat(paste("Running Simulation for ", simName, '\n',sep = ''))
   t1 <- Sys.time()
	
	C <- input_temp$VC
	K <- input_temp$FC
	B <- input_temp$B
	dprcn <-input_temp$Depreciation
	growth <- input_temp$Demand_Growth
	elst <- input_temp$Elasticity 
	T <- input_temp$Regulation_Length
	Distn <- input_temp$Distribution
	DistM <- input_temp$Mean
	DistSd <- input_temp$Standard.Dev.
	LB <- input_temp$LB
	UB <- input_temp$UB
	Margin <- input_temp$Margin
	PMax <- input_temp$PMax
	AssetType <- input_temp$Asset_Type
  Demand_Curve <- input_temp$Demand_Curve
	Surplus <- input_temp$Surplus
  Demand <- input_temp$Demand
  PR <- input_temp$PR
    sampN <- 1E6 # Number of simulation
  
  ## make 1 million draw from the distribution and only retain values between the LB and UB
  set.seed(1)
  

  AllSamp <- rnorm(sampN, DistM, DistSd)
  samps <- AllSamp[(AllSamp <= UB) & (AllSamp >=LB)]
  n <- length(samps)
  ## Get the percentiles
  ## Type = 1 makes sure the algorithm selects an observed value in the sample. In the case of "tie" it picks the smaller one. This guarantees that the percentile value matches the sample value exactly and speeds up calculation.  cf the cumulative sum bit for Cat 3
  pcts <- quantile(samps, probs = seq(0.01,1,0.01), type = 1) 
  pcts <- data.frame(
	  Pct=names(pcts),
	  PctVal=pcts,
	  stringsAsFactors=FALSE
	)
  
  samps <- data.frame(
    id = 1:length(samps),
    r = samps 
  )
  
  
  ## Find the sample that exactly matches the percentile value - so basically reducing the problem to
  ## 100 values rather than 1 million
  samps$Pct <- pcts[match(samps$r, pcts$PctVal), ]$Pct
  
  if(nrow(unique(samps[!is.na(samps$Pct), ])) != nrow(pcts)){
   stop('Percentile value cannot be matched in sample!')
  }
  
  ## calculate a bunch of intermediate step values
  # "expected discount rate" cf. eq 12-13   
  Edisct <- mean(1/(samps$r - growth)) 
  
  # Present value factor cf Eq 15
  samps$ET <- exp((growth-samps$r)*T)
  samps$PVF <- with(samps, ((1-ET)/(r-growth)+ET*Edisct))  
  
  ## Do the following if and only if it is a deferrable investment  
  if(AssetType == 'New'){
  # Discount rate used in Category 3 investment cf eq 31 -- calculated for every percentile
	samps$disct3 <- with(samps, exp(-r*T))  
  
	## calculate cumulative sum from Largest to smallest r
	samps <- samps[with(samps, order(r, decreasing  = TRUE)), ]
	samps$Disct3 <- with(samps, cumsum(disct3)) 
  
	Cat3Disct <- samps[!is.na(samps$Pct), c('Pct', 'r', 'Disct3')]
	Cat3Disct$Disct3 <- Cat3Disct$Disct3/nrow(samps) 
	}else{
	Cat3Disct <- NULL
	}

  
  EW <- ddply(pcts, c('Pct'),f_Cat, temp = samps, C=C, K=K, B=B, dprcn = dprcn, growth = growth, elst = elst, Margin = Margin, PMax = PMax, AssetType = AssetType, pct = pcts, Cat3Disct = Cat3Disct, Surplus = Surplus, Demand_Curve = Demand_Curve, Demand = Demand, PR = PR)
  t2 <- Sys.time()
  cat(paste("Done in", round(t2-t1, digits=2), "seconds\r\n"))
  return(EW)
  
  }
  
  
  f_Cat <- function(pct_r, temp, C, K, B, dprcn, growth, elst, Margin, PMax, pct, AssetType, Cat3Disct, Surplus, Demand_Curve, Demand, PR){ 
    
    #Evaluate welfare for each percentile 

    p_hat <- C+(pct_r$PctVal+dprcn)*K 

    p_retail <- p_hat + Margin # Add pass-through to obtain final retail price 
	  Q_0 <- B*(p_retail^elst) 
	
    # initial profitability cf eq9; Demand is now final retail demand
    prof_0 <- Q_0*(pct_r$PctVal-growth)*K  
	
    # Valuation of the firm cf eq14, evaluated for all sampled r values
    temp$Vf <- with(temp, PVF*prof_0 - K*Q_0) 
  
  if (Demand_Curve == 'CE') {
    CS <- -(B*p_retail^(elst+1))/(elst+1) 
    
  }else if (Demand_Curve == 'Isoelastic') {
    if (elst!=-1) {
      CS <- -(B*p_retail^(elst+1)-B*PMax^(elst+1))/(elst+1) 
    }
    else {
      CS <- -B*(log(p_retail)-log(PMax)) 
    }
  }else if (Demand_Curve == 'Linear') {
    a <- Demand*(1-elst)
    b <- elst*Demand/PR
    CS <- 0.5*(PMax - p_retail)*(a+b*p_retail)
    
  }else{
    stop(paste(Demand_Curve, " is not a defined demand curve! Simulation terminated", sep =''))
  }

    if(Surplus == 'Total'){ 
	  welfare_0 <- CS + prof_0
	}else if(Surplus == 'Consumer'){
	  welfare_0 <- CS
	}else{
	 stop(paste(Surplus, " is not a defined surplus measure! Simulation terminated", sep =''))
	}
    	
	if(AssetType == 'Existing'){ # Cat 1 investment
		temp$W <- with(temp, PVF*welfare_0 -K*Q_0)   
		EW <- mean(temp$W)
		}else if(AssetType == 'New'){ # Cat 3 investment
			## Find firm optimal investment threshold rc
		FirmVal <- temp[order(temp$r, decreasing = FALSE), ] 
		FirmVal$EVI <- cumsum(FirmVal$Vf) #
		FirmVal <- FirmVal[!is.na(FirmVal$Pct), ] 
		FirmVal$Disct3 <- Cat3Disct[match(FirmVal$Pct, Cat3Disct$Pct), ]$Disct3
		FirmVal$FirmVal <- with(FirmVal, EVI/(1-Disct3)) 
		
		rc_star <- FirmVal[FirmVal$FirmVal == max(FirmVal$FirmVal), ] 
		## Calculate Welfare
		temp3 <- temp[temp$r < rc_star$r, ] 
		temp3$W <- with(temp3, PVF*welfare_0 - K*Q_0) 
		EW <- ((1/nrow(temp))*sum(temp3$W))/(1-rc_star$Disct3)		
		
		}else{
		stop('AssetType not defined!')
		}
		
	return(data.frame(
		PctVal = pct_r$PctVal,
		AssetType = AssetType,
		EW = EW,
		stringsAsFactors = FALSE
	))		
  }



run <- function(InputSc) {
  
  cat(paste("Read input with", nrow(InputSc), "rows\r\n"))
  
  ## This is the actual code that does the simulation by calling the f_simulation function defined above.
  
  cat("Running preliminary data validation ...\r\n")
  InputSc <- InputSc[InputSc$Run == 'Yes', ]
  if(nrow(InputSc) < 2){
   stop("Simulation terminated. You need to enter at least one scenario.")
  }

    
  names(InputSc)[names(InputSc) == 'Current_Retail_Price'] <- 'PR'  

  
  ## Check all relevant data are numeric 
  
  datClass <- sapply(InputSc, class)
  datClass <- datClass[!names(datClass) %in% c("Scenario", "Demand_Curve", "Run", "Name", "Asset_Type", "Surplus")]
  
  errClass <- datClass[!datClass %in% c('numeric', 'integer')]
  
  if(length(errClass) > 0 ){
    stop("Simulation Terminated. Check you have entered numeric values in the required places.")  
  }
  
  ## Check numeric entries are positive/negative as they should be
  
  numCheck <- InputSc[, names(InputSc) %in% names(datClass)]
  
  if(any(numCheck$Elasticity >= 0)){
   stop("Simulation Terminated. Elasticity cannot be positive.") 
  }
  
  
  numCheck <- numCheck[,!names(numCheck) %in% c('Demand_Growth', 'Elasticity')]
  numCheckMax <- sapply(numCheck, max)
  
  ## Those entries must be strictly positive  
  if(any(numCheckMax[c('PR', 'PMax', 'Regulation_Length', 'Mean', 'Standard.Dev.')] <= 0)){  #DY removed PMax
   stop("Simulation Terminated. You must ensure that Price, Max Willingness to Pay, Regulation Length, Mean and Std of WACC are strictly positive entries.")  
  }
  
  ## Demand must be strictly positive for existing, can be 0 for new
  DemCheck <- InputSc[,c('Asset_Type', 'Demand')]
  
  if(any(DemCheck[DemCheck$Asset_Type == 'Existing', ]$Demand <=0)){
   stop("Simulation Terminated. Existing demand must be strictly positive.")
  }
  
  if(any(DemCheck[DemCheck$Asset_Type == 'New', ]$Demand <0)){
   stop("Simulation Terminated. New Demand cannot be negative. Check you have entered a non-negative proportion in the relevant entries.")   
  }
  
  ## Those entries must be non-negative
  
  if(any(numCheckMax[c('Depreciation')] < 0)){ #DY remove PTX, PDX, FX_TX and FX_DX
    stop("Simulation Terminated. You must ensure that Depreciation, Network cost as percentage of Bill and Network fixed cost components are non-negative entries.")  
   }  
   

  ##boundary at +- 4 standard deviation, but lower bound must be at least 0.5% above assumed demand growth   
  
   InputSc$UB <- with(InputSc, Mean + 4*Standard_Dev)
   InputSc$LB  <- with(InputSc, Mean - 4*Standard_Dev)
  
  ## Calculate "Margin" from transmission to retail bill, this effectively assumes a 1-to-1 pass-through of transmission price    
  InputSc$Margin <- with(InputSc, PR - Current_Input_Price) #DY don't need margin?

  # Demand function, initialise the parameter B as per in Dobbs paper
  InputSc$B <- with(InputSc, Demand/(PR^Elasticity))
  
   
  ## Create a temporary Id for each row of entry.
  InputSc$RowId <- 1:nrow(InputSc)
  
  ## Now run simulation and find out the Welfare by WACC percentile for each "Scenario - Asset Type" 
  ## ddply function from plyr package wrap around
  cat("Beginning simulations...\r\n")
  
  CatEW <- ddply(InputSc, c('RowId'), f_simulation)
   
  return(CatEW)
  
} 

 
  
  

